using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// This class represents VistALink protocol simple security response messages
	/// described in secSimpleResponse.xsd
	/// </summary>
	public abstract class SecuritySimpleResponseMessage : SecurityBaseCommentedResponseMessage
	{
		/// <summary>
		/// VistALink message XSD schema file name string. 		
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string XsdName = "secSimpleResponse.xsd";

		/// <summary>
		/// This constructor allows creating simple security response message
		/// with minimal set of parameters (omitting the optional informative message).
		/// </summary>
		/// <param name="responseType">Message response type. Required.</param>
		/// <param name="responseStatus">Message response status. Restricted to Success and Failure. Required.</param>
		protected SecuritySimpleResponseMessage( string responseType, ResponseStatus responseStatus )
			: this( responseType, responseStatus, null ) {}

		/// <summary>
		/// This constructor allows creating simple security response message with full 
		/// set of parameters including optional informative message. 
		/// </summary>
		/// <param name="responseType">Message response type. Required.</param>
		/// <param name="responseStatus">Message response status. Restricted to Success and Failure. Required.</param>
		/// <param name="message">Informative message. Optional.</param>
		protected SecuritySimpleResponseMessage( string responseType, ResponseStatus responseStatus, string message )
			: base( XsdName, responseType, responseStatus, message )
		{
			if( !IsValidResponseStatus( responseStatus ) )
				throw( new ArgumentOutOfRangeException( SR.Exceptions.SecSimpleResponseMessageWrongResponseStatus( responseStatus.ToString(), "SecuritySimpleResponseMessage" ) ) );
		}

		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="responseType">Message response type.</param>
		/// <param name="sourceDoc">Source XML document.</param>
		protected SecuritySimpleResponseMessage( string responseType, XmlDocument sourceDoc )
			: base( XsdName, responseType, sourceDoc )
		{
			if( !IsValidResponseStatus( _responseStatus ) )
				throw( new XmlParseException( SR.Exceptions.SecSimpleResponseMessageWrongResponseStatus( _responseStatus.ToString(), "SecuritySimpleResponseMessage" ) ) );
		}

		/// <summary>
		/// This method checks if supplied response status falls into acceptable range for
		/// simple response security message. 
		/// </summary>
		/// <param name="status">Response status to check.</param>
		/// <returns>True if response status is within acceptable range, false otherwise.</returns>
		private bool IsValidResponseStatus( ResponseStatus status )
		{
			return status == ResponseStatus.Success || status == ResponseStatus.Failure;
		}
	}
}
